<?php

/*
	SCHEDULER MEMBER CALL ORDER UTILITY API  CLASS

	CREATED ON : 06/09/2021 (d/m/yy)

    DETAILS : This is the main call order utility calss to control call order of the member to display 
              in scheduler or scheduler lead dashboard

              in a order of 
                A.callback date 
                B.contact number changes
                C.appoinment cancelled (appoinment cancelled,no show,assessment cancelled)
                D.newly assigned scheduler
                E.business day (not in current requirement list)
                Z.non call order members

              default call order json
              {
                  orderType=0,
                  sortOrder:"1602012010"
              }

              orderType - call order type 
              sortOrder - sort call order type by based on time updated value (unix time) 

*/

require_once("../TimeZone.php");
require_once("../utility/DataBaseUtility.php");
require_once("../utility/HraStatusUtility.php");
require_once("../report/ReportStatus.php");

class SchedulerCallOrderUtility
{
    //call order type
        const CALL_BACK_DATE_ORDER = 0;
        const CANCELLATION_ORDER = 1;
        const CONTACT_NO_CHANGED_ORDER = 2;
        const NEWLY_ASSIGNED_ORDER = 3;
        const BUSINESS_DAY_ORDER = 4;
    //used to set member with out call order
        const NON_CALL_ORDER = 50;

    /*
        NAME    : SetCallOrder
        TYPE    : global utiltiy function
        DETAILS : 
    */
    public static function setCallOrder($client_id,$year,$member_id,$call_order_type,$custom_data=null)
    {
         switch($call_order_type)
         {

            case self::CALL_BACK_DATE_ORDER:
                self::OnCallBackDateOrderAdded($client_id,$year,$member_id);
            break;
            
            case self::CONTACT_NO_CHANGED_ORDER:
                self::OnContactNumberOrderAdded($client_id,$year,$member_id);
            break;

            case self::CANCELLATION_ORDER:
                self::OnCancellationOrderAdded($client_id,$year,$member_id);
            break;
                
            case self::NEWLY_ASSIGNED_ORDER:
                self::OnNewlyAssignedSchedulerMembersAdded($client_id,$year,$member_id);
            break;

            case self::BUSINESS_DAY_ORDER: //business logic not current requirment list
                $sub_business_day_order = $custom_data['SUB_BUSINESS_ORDER'];
                self::OnBusinessDayOrderAdded($client_id,$year,$member_id,$sub_business_day_order);
            break;

            case self::NON_CALL_ORDER: //if no call order need clear that
                self::clearCallOrder($client_id,$year,$member_id);
            break;
         }
    }

     /*
        NAME    : setCallOrderWhenTrackerImport
        TYPE    : Internal Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function setCallOrderWhenTrackerImport($client_id,$year,$member_id,$report_status_text)
    {
        $report_status_text=trim($report_status_text);
        $report_status = ReportStatus::GetStringToStatus($report_status_text);
        switch($report_status)
        {
            case ReportStatus::CALLBACK_REQUESTED:
            case ReportStatus::F2F_CALLBACK:
                self::OnCallBackDateOrderAdded($client_id,$year,$member_id);
            break;

            case ReportStatus::CANCELLED:
                self::OnCancellationOrderImported($client_id,$year,$member_id);
            break;
        }
    }

    /*
        NAME    : OnCallBackDateOrderAdded
        TYPE    : Internal Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function clearCallOrder($client_id,$year,$member_id)
    {
        self::updateCallOrderInDb($client_id,$year,$member_id,NULL);
    }

    /*
        NAME    : OnCallBackDateOrderAdded
        TYPE    : Internal event function 
        DETAILS : when call back date is set in call log , only addedOn date is updated 
        RETURN  : 
    */
    private static function OnCallBackDateOrderAdded($client_id,$year,$member_id)
    {
        //when ever call back requested ,clear the scheduler call order because daily callback cron jobs will update the call
        //order when it due or over due
        self::clearCallOrder($client_id,$year,$member_id);
    }

    /*
        NAME    : OnContactNumberOrderAdded
        TYPE    : Internal event function 
        DETAILS : 
        RETURN  : 
    */
    private static function OnContactNumberOrderAdded($client_id,$year,$member_id)
    {
        $call_order_json = self::generateCallOrderAs(self::CONTACT_NO_CHANGED_ORDER,time());
        self::updateCallOrderInDb($client_id,$year,$member_id,$call_order_json);
    }

     /*
        NAME    : OnCancellationOrderAdded
        TYPE    : Internal event function 
        DETAILS : 
        RETURN  : 
    */
    private static function OnCancellationOrderAdded($client_id,$year,$member_id)
    {
        $call_order_json = self::generateCallOrderAs(self::CANCELLATION_ORDER,time());
        self::updateCallOrderInDb($client_id,$year,$member_id,$call_order_json);
    }

      /*
        NAME    : OnCancellationOrderImported
        TYPE    : Internal event function 
        DETAILS : this event called when every tracker staus of 'cancellation'
        RETURN  : 
    */
    private static function OnCancellationOrderImported($client_id,$year,$member_id)
    {
        $member_details = DataBaseUtility::getMemberDataBaseFields($client_id,$year,$member_id,['hraStatus']);
        if($member_details)
        {
            $hra_status_array = json_decode($member_details["hraStatus"],true);
            if($hra_status_array && array_key_exists('hraStatus',$hra_status_array))
            {
                $hra_status = $hra_status_array['hraStatus'];
                switch($hra_status)
                {
                    case MemberUtility::PATIENT_REFUSAL:
                    case MemberUtility::APPOINTMENT_CANCELLED:
                    case MemberUtility::ASSESSMENT_CANCELLED:
                    case MemberUtility::NO_SHOW:
                        $hra_status_history_current_date = HraStatusUtility::getCurrentHraStatusHistoryDate($client_id,$year,$member_id,$hra_status);
                        if($hra_status_history_current_date)
                        {
                            $converted_date = DateUtility::splitAndJoinDelimiterDate($hra_status_history_current_date);
                            if($converted_date)
                            {
                                $time_stamp = strtotime($converted_date);
                                if($time_stamp)
                                {
                                    $call_order_json = self::generateCallOrderAs(self::CANCELLATION_ORDER,strtotime($converted_date));
                                    self::updateCallOrderInDb($client_id,$year,$member_id,$call_order_json);
                                }
                            }
                        }
                    break;
                }
            }
        }
        
    }

    /*
        NAME    : OnNewlyAssignedSchedulerMembersAdded
        TYPE    : Internal event function 
        DETAILS : 
        RETURN  : 
    */
    private static function OnNewlyAssignedSchedulerMembersAdded($client_id,$year,$member_id)
    {
        $call_order_json = self::generateCallOrderAs(self::NEWLY_ASSIGNED_ORDER,time());
        self::updateCallOrderInDb($client_id,$year,$member_id,$call_order_json);
    }

    /*
        NAME    : OnBusinessDayOrderAdded
        TYPE    : Internal event function 
        DETAILS :  5,7,7 day business logic [not in requirement list in current phase ]
        RETURN  : 
    */
    private static function OnBusinessDayOrderAdded($client_id,$year,$member_id,$sub_business_day_order)
    {
        $call_order_json = self::generateCallOrderAs(self::NEWLY_ASSIGNED_ORDER,time());

    }

     /*
        NAME    : OnCallBackDateOrderAdded
        TYPE    : Internal tiltiy function 
        DETAILS : 
        RETURN  : 
    */
    private static function generateCallOrderAs($call_order_type,$call_order_sort_value,$custom_data=null,$is_json=true)
    {
        $call_order = array(
            'orderType'=>$call_order_type,
            'sortOrder'=>$call_order_sort_value
        );

        switch($call_order_type)
        {
            case self::CALL_BACK_DATE_ORDER:
                if(array_key_exists('callBackDate',$custom_data))
                {
                    $call_order['callBackDate']=$custom_data['callBackDate'];
                }
            break;
        }

        if($is_json)
        {
            return json_encode($call_order);
        }

        return $call_order;
    }

    
    /*
        NAME    : updateCallOrderInDb
        TYPE    : Internal Utiltiy database function 
        DETAILS : 
        RETURN  : 
    */
    public static function updateCallOrderInDb($client_id,$year,$member_id,$call_order)
    {
        //database connection
        include("../db.php");

            $table_name = "HRA_member_".$client_id.'_'.$year;
            $call_order_update_query=$conn->prepare('UPDATE '.$table_name.' SET schedulerCallOrder = ? WHERE member_id = ?');
            $call_order_update_query->bind_param('ss',$call_order,$member_id);
            $call_order_update_query->execute();
        
        return $call_order_update_query->affected_rows;
    }

    /*
        NAME    : updateCallOrderInDbWhenNull
        TYPE    : Internal Utiltiy database function 
        DETAILS : 
        RETURN  : 
    */
    public static function updateCallOrderInDbWhenNull($client_id,$year,$member_id,$call_order)
    {
        //database connection
        include("../db.php");

            $table_name = "HRA_member_".$client_id.'_'.$year;
            $call_order_update_query=$conn->prepare('UPDATE '.$table_name.' SET schedulerCallOrder = ? WHERE member_id = ? AND schedulerCallOrder IS NULL');
            $call_order_update_query->bind_param('ss',$call_order,$member_id);
            $call_order_update_query->execute();
        
        return $call_order_update_query->affected_rows;
    }

}


?>