<?php

/*
    GLOBAL DATABASE UTILITY FUNCTIONS
*/

require_once("../TimeZone.php");
require_once("StringUtility.php");


class DataBaseUtility
{

     /*
        NAME    : isTableAvailable
        TYPE    : Utiltiy function 
        DETAILS : check table present or not to avoid runtime error
        RETURN  : if table present true else false
    */
    public static function isTableAvailable($client_id,$year)
    {
            //database connection
            include("../db.php");
            //first check next year table already created or not
            $table_name = "HRA_member_".$client_id.'_'.$year;
            $check_table_query="SHOW TABLES LIKE '".$table_name."'";
            $check_table_query_result=mysqli_query($conn,$check_table_query);
            //if sql query result with zero means no such a table
            if(mysqli_num_rows($check_table_query_result) == 1)
            {
                return true;
            }

            return false;
    }

    /*
        NAME    : IsPlanAvailable
        TYPE    : Utiltiy function 
        DETAILS : check whether the client plan already available or not 
        RETURN  : if present array else NULL
    */
    public static function IsPlanAvailable($plan_name)
    {
        //database connection
        include("../db.php");

            $client_plan_query=$conn->prepare('SELECT plan from HRA_client WHERE plan=?');
            $client_plan_query->bind_param('s',$plan_name);
            $client_plan_query->execute();
            $client_plan_result = $client_plan_query->get_result();
            $client_plan_details=$client_plan_result->fetch_assoc();

        return $client_plan_details;
    }

     /*
        NAME    : IsClientAvaiable
        TYPE    : Utiltiy function 
        DETAILS : check whether the client year already available or not 
        RETURN  : if present true else false
    */
    public static function IsClientAvaiable($plan_name,$program_name,$client_id)
    {
        //database connection
        include("../db.php");

            $client_query=$conn->prepare('SELECT * from HRA_client WHERE (plan=? and program=?) or clientId=?');
            $client_query->bind_param('sss',$plan_name,$program_name,$client_id);
            $client_query->execute();
            $client_result = $client_query->get_result();
            $client_details=$client_result->fetch_assoc();

        return $client_details;
    }

     /*
        NAME    : InsertNewClient
        TYPE    : global Utiltiy function 
        DETAILS : insert the new client data .year will be reamove in future
        RETURN  :  afftected row count
    */
    public static function InsertNewClient($plan_name,$program_name,$client_name,$health_plan_name,$client_id,$year,$client_info,$created_on,$updated_on)
    {
        //database connection
        include("../db.php");

         //if no program name set consider no plan type 
         $plan_type = null;
         $program_name_is_none = strtolower(trim($program_name));
         if(StringUtility::isValueSet($program_name) && $program_name_is_none!='none')
         {
             $plan_type='program';
         }
         else
         {
            $program_name='none';
         }

        $client_insert_query = $conn->prepare('INSERT INTO HRA_client (plan,plan_type,program,client_name,health_plan_name,clientId,year,clientInfo,createdon,updatedon) 
                                VALUES (?,?,?,?,?,?,?,?,?,?)');
        $client_insert_query->bind_param("ssssssssss", $plan_name,$plan_type,$program_name,$client_name,$health_plan_name,$client_id,$year,$client_info,$created_on,$updated_on);
        $client_insert_query->execute();

        return $client_insert_query->affected_rows;
    }

    /*
        NAME    : updateClient
        TYPE    : global Utiltiy function 
        DETAILS : update client data
        RETURN  :  afftected row count
    */
    public static function updateClient($plan_name,$program_name,$client_name,$health_plan_name,$client_id,$year,$client_info,$updated_on)
    {
        //database connection
        include("../db.php");
        //UPDATE user_privileges set client_privileges=? WHERE user_unique_id=?
        //if no program name set consider no plan type 
        $plan_type = null;
        $program_name_is_none = strtolower(trim($program_name));
        if(StringUtility::isValueSet($program_name) && $program_name_is_none!='none')
        {
            $plan_type='program';
        }
        else
        {
            $program_name='none';
        }
        
        $client_insert_query = $conn->prepare('UPDATE HRA_client SET plan=?,plan_type=?,program=?,client_name=?,health_plan_name=?,year=?,clientInfo=?,updatedon=? WHERE clientId=?'); 
        $client_insert_query->bind_param("sssssssss", $plan_name,$plan_type,$program_name,$client_name,$health_plan_name,$year,$client_info,$updated_on,$client_id);
        $client_insert_query->execute();

        return $client_insert_query->affected_rows;
    }

    /*
        NAME    : updateClientWithoutYear
        TYPE    : global Utiltiy function 
        DETAILS : update client data
        RETURN  :  afftected row count
    */
    public static function updateClientWithoutYear($plan_name,$program_name,$client_name,$health_plan_name,$client_id,$client_info,$updated_on)
    {
        //database connection
        include("../db.php");
        //UPDATE user_privileges set client_privileges=? WHERE user_unique_id=?
        //if no program name set consider no plan type 
        $plan_type = null;
        $program_name_is_none = strtolower(trim($program_name));
        if(StringUtility::isValueSet($program_name) && $program_name_is_none!='none')
        {
            $plan_type='program';
        }
        else
        {
            $program_name='none';
        }

        $client_insert_query = $conn->prepare('UPDATE HRA_client SET plan=?,plan_type=?,program=?,client_name=?,health_plan_name=?,clientInfo=?,updatedon=? WHERE clientId=?'); 
        $client_insert_query->bind_param("ssssssss", $plan_name,$plan_type,$program_name,$client_name,$health_plan_name,$client_info,$updated_on,$client_id);
        $client_insert_query->execute();

        return $client_insert_query->affected_rows;
    }

    /*
        NAME    : IsClientYearAvaiable
        TYPE    : Utiltiy function 
        DETAILS : check whether the client year already available or not 
        RETURN  : if present true else false
    */
    public static function IsClientYearAvailable($client_id,$year)
    {
        //database connection
        include("../db.php");

            $client_year_query=$conn->prepare('SELECT * from hra_client_year_details WHERE client_id=? and year=?');
            $client_year_query->bind_param('si',$client_id,$year);
            $client_year_query->execute();
            $client_year_result = $client_year_query->get_result();
            $client_year_details=$client_year_result->fetch_assoc();

        return $client_year_details;
    }

     /*
        NAME    : AddNewClientYear
        TYPE    : global Utiltiy function 
        DETAILS : client and year set as unique constraint ...
        RETURN  : 
    */
    public static function AddNewClientYear($client_id,$client_year)
    {
        //database connection
        include("../db.php");
            //in table hra client year -> client_id_year_unique indexes used to check unique client_id and year when new year inserted
            $client_insert_query = $conn->prepare('INSERT IGNORE INTO hra_client_year_details (client_id,year) VALUES (?,?)');                       
            $client_insert_query->bind_param("si",$client_id,$client_year);
            $client_insert_query->execute();

        return $client_insert_query->affected_rows;
    }


    /*
        NAME    : IsClientLobAvailable
        TYPE    : Utiltiy function 
        DETAILS : check whether the client year already available or not 
        RETURN  : if present true else false
    */
    public static function IsClientLobAvailable($client_id,$year,$lob)
    {
        //database connection
        include("../db.php");

        $client_lob_query=$conn->prepare('SELECT * from hra_client_lob_details WHERE client_id=? and year=? and lob=?');
        $client_lob_query->bind_param('sis',$client_id,$year,$lob);
        $client_lob_query->execute();
        $client_lob_result = $client_lob_query->get_result();
        $client_lob_details=$client_lob_result->fetch_assoc();

        return $client_lob_details;
    }

    /*
        NAME    : AddLobIfNotAvailable
        TYPE    : global Utiltiy function 
        DETAILS : check whether the client year already available or not 
        RETURN  : if present true else false
    */
    public static function AddLobIfNotAvailable($client_id,$year,$lob)
    {
        //database connection
        include("../db.php");
        //check for already client_id,year,lob
        //note:table already have constraint 
        if(self::IsClientLobAvailable($client_id,$year,$lob)==null)
        {
            $lob_upper_case = strtoupper($lob);
            $client_insert_query = $conn->prepare('INSERT IGNORE INTO hra_client_lob_details (client_id,year,lob) 
                                    VALUES (?,?,?)');
            $client_insert_query->bind_param("sis",$client_id,$year,$lob_upper_case);
            $client_insert_query->execute();

            return $client_insert_query->affected_rows;
        }

        return 0;
    }


  public static function getAllLobsListByYear($planId,$year){
        include("../db.php");
        $all_client_details=$conn->prepare('SELECT clientId,program FROM HRA_client WHERE plan=? and year like ?');
        $year1='%'.$year.'%';
        $all_client_details->bind_param('ss',$planId,$year1);
        $all_client_details->execute();
        $all_clients_result = $all_client_details->get_result();
        $all_clients_result_details=[];
        $lobdatabyprogram = [];
        $all_lob_details=[];

        while ($client_row = $all_clients_result->fetch_assoc()) {
            $clientId=$client_row['clientId'];
            $program=$client_row['program'];

$lobdatabyprogram['clientId'] = $clientId;

            $lobdatabyprogram['program'] = $program;
            $member_records_query = "SELECT DISTINCT trim(lob) as lob,client_id FROM hra_client_lob_details where client_id = '$clientId' and year = '$year'  ORDER BY lob ";

            $member_records_query_result=mysqli_query($conn,$member_records_query);

           $insidearay = [];
            while($getVersionresultrow=mysqli_fetch_array($member_records_query_result,MYSQLI_ASSOC)){
               $insidearay[] = $getVersionresultrow['lob'];  
            }
            $lobdatabyprogram['lob']=$insidearay;
array_push($all_lob_details,$lobdatabyprogram);
        }
return $all_lob_details;
    }

     /*
        NAME    : getAllLobs
        TYPE    : global Utiltiy function 
        DETAILS : get the all lobs irrespective of client
        RETURN  : array of available lobs
    */
  public static function getAllLobs($planId)
    {
        //database connection
        include("../db.php");
        $all_client_details=$conn->prepare('SELECT clientId FROM HRA_client WHERE plan=?');
        $all_client_details->bind_param('s',$planId);
        $all_client_details->execute();
        $all_clients_result = $all_client_details->get_result();
        $all_clients_result_details=[];
        while ($client_row = $all_clients_result->fetch_assoc()) {
            $all_clients_result_details[]=$client_row['clientId'];
        }
         $clientslist = "'" . implode ( "', '", $all_clients_result_details ) . "'";
		$member_records_query = "SELECT DISTINCT trim(lob) as lob FROM hra_client_lob_details where client_id in (".$clientslist.") ORDER BY lob ";

		$member_records_query_result=mysqli_query($conn,$member_records_query);
		$all_lob_details=[];
		while($getVersionresultrow=mysqli_fetch_array($member_records_query_result,MYSQLI_ASSOC)){
		$all_lob_details[]=$getVersionresultrow['lob'];
		}
		return $all_lob_details;
    }

     /*
        NAME    : getClientLobs
        TYPE    : global Utiltiy function 
        DETAILS : get the all lobs based on client id
        RETURN  : array of available lobs
    */
    public static function getClientLobs($client_id)
    {

         //database connection
         include("../db.php");

            $client_lob_query=$conn->prepare('SELECT lob from hra_client_lob_details WHERE client_id=?');
            $client_lob_query->bind_param('s',$client_id);
            $client_lob_query->execute();
            $client_lob_result = $client_lob_query->get_result();

            $client_lob_details=[];
            while ($lob_row = $client_lob_result->fetch_assoc()) {
                $client_lob_details[]=$lob_row['lob'];
            }
            return array_unique($client_lob_details);

    }

    /*
        NAME    : getClientYearLobs
        TYPE    : global Utiltiy function 
        DETAILS : get the all lobs based on client id and year
        RETURN  : array of available lobs
    */
    public static function getClientYearLobs($client_id,$year)
    {

         //database connection
         include("../db.php");

            $client_lob_query=$conn->prepare('SELECT lob from hra_client_lob_details WHERE client_id=? and year=?');
            $client_lob_query->bind_param('si',$client_id,$year);
            $client_lob_query->execute();
            $client_lob_result = $client_lob_query->get_result();

            $client_lob_details=[];
            while ($lob_row = $client_lob_result->fetch_assoc()) {

                $client_lob_details[]=$lob_row['lob'];
            }
            
            return $client_lob_details;

    }

    /*
        NAME    : getClientYears
        TYPE    : global Utiltiy function 
        DETAILS : get the all lobs based on client id
        RETURN  : array of available years
    */
    public static function getClientYears($client_id)
    {

         //database connection
         include("../db.php");

            $client_year_query=$conn->prepare('SELECT year from hra_client_year_details WHERE client_id=?');
            $client_year_query->bind_param('s',$client_id);
            $client_year_query->execute();
            $client_year_result = $client_year_query->get_result();
            $client_year_details = null;
            while ($year_result_row = $client_year_result->fetch_assoc()) {

                $client_year_details[]=$year_result_row['year'];
            }
            
        return $client_year_details!=null?array_unique($client_year_details):null;

    }

    /*
        NAME    : IsUserPrivillegeAvailable
        TYPE    : global Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function IsUserPrivilegeAvailable($user_unique_id,$name_of_privilege)
    {
        //database connection
        include("../db.php");

            $user_client_privg_query=$conn->prepare('SELECT * from user_privileges WHERE user_unique_id=?');
            $user_client_privg_query->bind_param('i',$user_unique_id);
            $user_client_privg_query->execute();
            $user_client_privg_query_result = $user_client_privg_query->get_result();
            $user_client_privillege=$user_client_privg_query_result->fetch_assoc();
        
            if(is_array($user_client_privillege) && array_key_exists($name_of_privilege,$user_client_privillege))
            {
                return $user_client_privillege[$name_of_privilege];
            }
            
            return null;
    }

     /*
        NAME    : updateOrInsertUserClientPrivilege
        TYPE    : global Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function updateOrInsertUserClientPrivilege($user_unique_id,$client_privileges)
    {
        //database connection
        include("../db.php");

        $user_status = false;
        $affected_rows = 0;

        $user_details = self::getUserById($user_unique_id);
        if($user_details)
        {
            if(self::IsUserPrivilegeAvailable($user_unique_id,'client_privileges'))
            {
                $user_update_query=$conn->prepare('UPDATE user_privileges set client_privileges=? WHERE user_unique_id=?');
                $user_update_query->bind_param('si',$client_privileges,$user_unique_id);
                $user_update_query_result=$user_update_query->execute();
                $user_status = $user_update_query_result;
                $affected_rows = $user_update_query->affected_rows;
            }
            else
            {
                $user_client_privg_insert_query = $conn->prepare('INSERT INTO user_privileges (client_privileges,user_unique_id) VALUES (?,?)');
                $user_client_privg_insert_query->bind_param("si",$client_privileges,$user_unique_id);
                $user_insert_result=$user_client_privg_insert_query->execute();
                $user_status = $user_insert_result;
                $affected_rows = $user_client_privg_insert_query->affected_rows;
            }
        }

        return array("status"=>$user_status,"rowsUpdated"=>$affected_rows);

    }

    /*
        NAME    : getUserClientPrivillege
        TYPE    : global Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function getUserClientPrivilege($user_unique_id)
    {

         //database connection
         include("../db.php");

            $user_client_privileges=$conn->prepare('SELECT client_privileges from user_privileges WHERE user_unique_id=?');
            $user_client_privileges->bind_param('i',$user_unique_id);
            $user_client_privileges->execute();
            $user_client_privileges_result = $user_client_privileges->get_result();
            $user_client_array = $user_client_privileges_result->fetch_assoc();
            if($user_client_array)
            {
                $user_client_privileges_array = json_decode($user_client_array['client_privileges'],true);
                if($user_client_privileges_array && is_array($user_client_privileges_array) && array_key_exists("clients",$user_client_privileges_array))
                {
                    return $user_client_privileges_array["clients"];
                }
            }

         return null;

    }

    /*
        NAME    : getUserClientPrivillege
        TYPE    : global Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function getUserClientPrivilegeByPlan($plan_id,$user_unique_id)
    {

         //database connection
         include("../db.php");

            $user_client_privileges=$conn->prepare('SELECT
                                        client_table.clientId as client_id
                                    FROM
                                        (
                                        SELECT
                                            JSON_EXTRACT(client_privileges, "$.clients") AS user_client_id
                                        FROM
                                            user_privileges
                                        WHERE
                                            user_unique_id = ?
                                    ) AS user_privileges_table
                                    LEFT JOIN HRA_client AS client_table
                                    ON
                                        client_table.clientId =
                                            user_privileges_table.user_client_id
                                         WHERE client_table.plan = ?');
            $user_client_privileges->bind_param('is',$user_unique_id,$plan_id);
            $user_client_privileges->execute();
            $user_client_privileges_result = $user_client_privileges->get_result();

            while($user_client_row=mysqli_fetch_array($user_client_privileges_result,MYSQLI_ASSOC))
			{
				$user_client_list[]=$user_client_row['client_id'];
			}
           
            return $user_client_list;
            
    }


    

     /*
        NAME    : isMemberAvailable
        TYPE    : Utiltiy function 
        DETAILS : check whether member avaialble or not
        RETURN  : if table present true else false
    */
    public static function isMemberAvailable($client_id,$year,$member_id)
    {
        //database connection
        include("../db.php");

		$member_sql_query = "SELECT member_id FROM HRA_member_".$client_id.'_'.$year." WHERE member_id='$member_id'";
		$db_success = false;
		try
		{
			$member_sql_query_result=mysqli_query($conn,$member_sql_query);
            if($member_sql_query_result)
            {
                if(mysqli_num_rows($member_sql_query_result) == 1)
                {
                    $db_success=true;
                }
            }
		}
		catch (Exception $e)
		{
			$error = $e->getMessage();
			$db_success=false;
		}

        return $db_success;
    }

    /*
		NAME 	: CreateMemberTable
		TYPE 	: global Utility Function
        DETAILS	: create member table based on client id and year
        RETURN	: true if success else false   
	*/
    public static function CreateMemberTable($client_id,$year)
    {
        //database connection
		include("../db.php");

        $create_member_table_query = "CREATE TABLE HRA_member_" . $client_id . "_" . $year . " (
                id INT(11) AUTO_INCREMENT PRIMARY KEY,
                qa_id INT(11) NULL,
                coder_id INT(11) NULL,
                nurse_id VARCHAR(100) NULL,
                member_id VARCHAR(100) NOT NULL,
                member_id_two VARCHAR(100) NULL,
                member_unique_id VARCHAR(100)  NULL,
                scheduler_id VARCHAR(100)  NULL,
                firstname VARCHAR(200)  NULL,
                lastname VARCHAR(200)  NULL,
                hicn VARCHAR(200)  NULL,
                memberinfo JSON  NULL,
                clientid VARCHAR(50)  NULL,
                year VARCHAR(200)  NULL,
                hra VARCHAR(200)  NULL,
                hraStatus TEXT  NULL,
                reportStatus JSON NULL DEFAULT NULL,
                currentStatus JSON NULL DEFAULT NULL,
                reportStatusHistory json DEFAULT NULL,
                reportHraStatusWeekly json DEFAULT NULL,
                hraStatusHistory JSON NULL,
                hraFileUpdatedHistory JSON NULL,
                cancelHistory JSON NULL,
                versioning TEXT  NULL,
                query_form  JSON  NULL,
                scheduler  JSON  NULL,
                schedulerDetails JSON NULL,
                schedulerCallHraStatus JSON NULL,
                schedulerCallLogDetails JSON NULL,
                schedulerCallOrder JSON NULL DEFAULT NULL,
                availForSchedule tinyint(1) NOT NULL DEFAULT 1,
                importTrackerDetails JSON NULL DEFAULT NULL,
                pcp_fax_response JSON NULL DEFAULT NULL,
                VisitType VARCHAR(200)  NULL,
                scheduleDate VARCHAR(200)  NULL,
                termDate VARCHAR(200)  NULL,
                EvaluationDate VARCHAR(200)  NULL,
                refusalLetterDate TIMESTAMP NULL DEFAULT NULL,
                sentToAvility VARCHAR(200) NULL,
                invoiceDate VARCHAR(200) NULL,
                reportNotes JSON NULL,
                admin_idno VARCHAR(200)  NULL,
                created_on VARCHAR(200)  NULL,
                updated_on VARCHAR(200)  NULL,
                isDownload bit(1) DEFAULT NULL,	
                DownloadDate date DEFAULT NULL,	
                utcfiles text,	
                Ftp_upload_status json DEFAULT NULL,	
                pdf_file_name varchar(100) DEFAULT NULL,	
                client_ftp_upload_status json DEFAULT NULL,	
                summary_pdf_file_name varchar(100) DEFAULT NULL,	
                referralForm json DEFAULT NULL,	
                clinician_signature JSON  NULL,	
                exam_finished_date varchar(200) DEFAULT NULL,	
                member_pdf_file_name varchar(50) DEFAULT NULL,	
                member_ftp_upload_status json DEFAULT NULL,	
                utcLetterDownloadDate timestamp NULL DEFAULT NULL,	
                declineLetterDownloadDate timestamp NULL DEFAULT NULL,
                addendum_clinician_signature JSON NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                )";

        if ($conn->query($create_member_table_query) === TRUE) 
        {
            return true;
        }

        return false;

    }


    /*
		NAME 	: getClientsList
		TYPE 	: global Utility Function
        DETAILS	: get alll client list as array
        RETURN	: client list array    
	*/
	public static function getClientsList()
	{	
		//database connection
		include("../db.php");
			$client_list = NULL;
			//sql query to get all client list 
			$client_list_query = "SELECT * from HRA_client";
			$client_list_query_result=mysqli_query($conn,$client_list_query);
			//fetch all the client info
			while($client_info_row=mysqli_fetch_array($client_list_query_result,MYSQLI_ASSOC))
			{
				$client_list[]=$client_info_row;
			}

		return $client_list; 
	}


    /*
		NAME 	: getClientIdList
		TYPE 	: global Utility Function
        DETAILS	: get all client id  as array
        RETURN	: client list array    
	*/
    public static function getClientIdList()
	{	
		//database connection
		include("../db.php");
			$client_id_list = NULL;
			//sql query to get all client list 
			$client_list_query = "SELECT clientId from HRA_client";
			$client_list_query_result=mysqli_query($conn,$client_list_query);
			//fetch all the client info
			while($client_info_row=mysqli_fetch_array($client_list_query_result,MYSQLI_ASSOC))
			{
				$client_id_list[]=$client_info_row["clientId"];
			}

		return $client_id_list; 
	}

    /*
		NAME 	: getClientIdListByPlan
		TYPE 	: global Utility Function
        DETAILS	: get all client id  as array
        RETURN	: client list array    
	*/
    public static function getClientIdListByPlan($plan_name)
	{	
		//database connection
		include("../db.php");
			$client_id_list = NULL;
			//sql query to get all client list 
			$client_list_query = $conn->prepare('SELECT clientId from HRA_client WHERE plan=?');
            $client_list_query->bind_param('s',$plan_name);
            $client_list_query->execute();
            $client_query_result = $client_list_query->get_result();

            while($client_info_row=$client_query_result->fetch_assoc())
			{
				$client_id_list[]=$client_info_row["clientId"];
			}
			//fetch all the client info
			
		return $client_id_list; 
	}

    
    /*
		NAME 	: getClientName
		TYPE 	: global Utility Function
        DETAILS	: get all client list as array
        RETURN	: client list array    
	*/
    public static function getClientName($client_id)
	{	
		//database connection
		include("../db.php");

            $client_query=$conn->prepare('SELECT client_name from HRA_client where clientId = ?');
            $client_query->bind_param('s',$client_id);
            $client_query->execute();
            $client_query_result = $client_query->get_result();
            $client_detail_array=$client_query_result->fetch_assoc();

        if($client_detail_array)
        {
            return  $client_detail_array['client_name'];
        }

		return 'None'; 
	}

    public static function getPlanName($client_id)
	{	
		//database connection
		include("../db.php");

            $client_query=$conn->prepare('SELECT plan from HRA_client where clientId = ?');
            $client_query->bind_param('s',$client_id);
            $client_query->execute();
            $client_query_result = $client_query->get_result();
            $client_detail_array=$client_query_result->fetch_assoc();

        if($client_detail_array)
        {
            return  $client_detail_array['plan'];
        }

		return 'None'; 
	}

    /*
		NAME 	: runSqlQueryOnAllClientYears
		TYPE 	: global  utility callable Function
        DETAILS	: 
        RETURN	:    
	*/
    public static function runSqlQueryOnAllClientYears(callable $func_callable)
    {
        $client_list = self::getClientsList();
		//total client count 
		$client_count = count($client_list);
		for($clt=0;$clt<$client_count;$clt++)
		{
			//read the client array
			$client = $client_list[$clt];
			$client_id = $client['clientId'];
			$client_name = $client['client_name'];

			//parse the years from year string column
			$client_years =$client['year'];
			$client_year_array = explode(",",$client_years);
			$client_years_count = count($client_year_array);

			for($y=0;$y<$client_years_count;$y++)
			{
                $func_callable($client_id,$client_year_array[$y]);
            }
        }
    }

    
    /*
		NAME 	: runSqlQueryOnClientYears
		TYPE 	: global  utility callable Function
        DETAILS	: 
        RETURN	:    
	*/
    public static function runSqlQueryOnClientYears($client_id,callable $func_callable)
    {
        $client_list = self::getClientsList();
		//total client count 
		$client_count = count($client_list);
		for($clt=0;$clt<$client_count;$clt++)
		{
			//read the client array
			$client = $client_list[$clt];

            if($client_id== $client['clientId'])
            {
                $client_name = $client['client_name'];
                //parse the years from year string column
                $client_years =$client['year'];
                $client_year_array = explode(",",$client_years);
                $client_years_count = count($client_year_array);

                for($y=0;$y<$client_years_count;$y++)
                {
                    $func_callable($client_id,$client_year_array[$y]);
                }
            }
        }
    }

    /*
        NAME    : resetUserPassword
        TYPE    : Internal Utiltiy function 
        DETAILS : 
        RETURN  : 
    */
    public static function resetUserPassword($user_name,$new_password)
    {
        //database connection
        include("../db.php");

        $password_hash = Utility::generatePasswordHash($new_password);
        $user_update_query=$conn->prepare('UPDATE userLogin SET password_hash = ? WHERE BINARY username = BINARY ?');
        $user_update_query->bind_param('ss',$password_hash,$user_name);
        $user_update_query->execute();
        
        return $user_update_query->affected_rows;
        
    }

    /*
        NAME    : getUserBygetUserByIdEmail
        TYPE    : Internal Utiltiy function 
        DETAILS : get user details by user id
        RETURN  : user details array 
    */
    public static function getUserById($user_id)
    {
        //database connection
        include("../db.php");

        $user_query=$conn->prepare('SELECT type,username,firstname,lastname,email,userStatus FROM `userLogin` WHERE `userUniqueno`= ?');
        $user_query->bind_param('i',$user_id);
        $user_query->execute();
        $user_query_result = $user_query->get_result();
        $user_detail_array=$user_query_result->fetch_assoc();
        if($user_detail_array)
        {
            return $user_detail_array;
        }

        //user not found
        return null;
       
    }

    /*
        NAME    : getUserIdByname
        TYPE    : global Utiltiy function 
        DETAILS : get user details by user name
        RETURN  : user details array 
    */
    public static function getUserIdByUserName($user_name)
    {
       //database connection
       include("../db.php");
        //user data query
        $user_query=$conn->prepare('SELECT userUniqueno,type,username,firstname,lastname,email,userStatus FROM userLogin WHERE BINARY username = BINARY ?');
            $user_query->bind_param('s',$user_name);
            $user_query->execute();
            $user_query_result = $user_query->get_result();
            $user_detail_array=$user_query_result->fetch_assoc();
            
        if($user_detail_array)
        {
            return $user_detail_array;
        }

        //user not found
        return null;
    }

    /*
        NAME    : getUserByEmailAndRole
        TYPE    : global Utiltiy function 
        DETAILS : get user details by email and role
        RETURN  : user details array 
    */
    public static function getUserByEmailAndRole($user_email,$user_role)
    {
        //database connection
        include("../db.php");

        $user_query=$conn->prepare('SELECT userUniqueno,username,firstname,lastname,email,userStatus FROM userLogin WHERE email = ? and type = ?');
        $user_query->bind_param('ss',$user_email,$user_role);
        $user_query->execute();
        $user_query_result = $user_query->get_result();
        $user_detail_array=$user_query_result->fetch_assoc();
        if($user_detail_array)
        {
            return $user_detail_array;
        }

        //user not found
        return null;
       
    }

    /*
        NAME    : getUsersByEmail
        TYPE    : global Utiltiy function 
        DETAILS : get users details by same email  but different  username and role
        RETURN  : user details array 
    */
    public static function getUsersByEmail($user_email)
    {
        //database connection
        include("../db.php");

        $user_query=$conn->prepare('SELECT type,userUniqueno,username,firstname,lastname,userStatus FROM userLogin WHERE email = ?');
        $user_query->bind_param('s',$user_email);
        $user_query->execute();
        $user_query_result = $user_query->get_result();

        $user_detail_array =[];

        while ($user_detail = $user_query_result->fetch_assoc()) {
            $user_detail_array[]=$user_detail;
        }

        return $user_detail_array;
       
    }


        /*
            NAME    : getUserIdByFullName
            TYPE    : global Utiltiy function 
            DETAILS : get user details by first name and last name
            RETURN  : user details array 
        */
        public static function getUserIdByFullName($firstname,$lastname)
        {

            //database connection
            include("../db.php");

            //user data query
            $user_data_query = "SELECT userUniqueno,type,firstname,lastname,email,userStatus FROM `userLogin` WHERE `firstname`= '$firstname' and `lastname`= '$lastname'";
            try 
            {
                $user_data = null;
                $user_data_query_result=mysqli_query($conn,$user_data_query);
                if(mysqli_num_rows($user_data_query_result) == 1)
                {
                    $user_data_row=mysqli_fetch_array($user_data_query_result,MYSQLI_ASSOC);
                    $user_data = $user_data_row;
                    return $user_data;
                }          
            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
                return null;
            }

            //user not found
            return null;
        }

        /*
            NAME    : getUserDetailsByFirstNameAndType
            TYPE    : global Utiltiy function 
            DETAILS : get user details by first name and type
            RETURN  : user details array 
        */
        public static function getUserDetailsByFirstNameAndType($firstname,$type)
        {

            //database connection
            include("../db.php");

            //user data query
            $user_data_query = "SELECT userUniqueno,username,type,firstname,lastname,email,userStatus FROM `userLogin` WHERE `firstname`= '$firstname' and `type`= '$type'";
            try 
            {
                $user_data = null;
                $user_data_query_result=mysqli_query($conn,$user_data_query);
                if(mysqli_num_rows($user_data_query_result) >= 1)
                {
                    $user_data_row=mysqli_fetch_array($user_data_query_result,MYSQLI_ASSOC);
                    $user_data = $user_data_row;
                    return $user_data;
                }            

            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
                return null;
            }

            //user not found
            return null;
        }

        public static function getUserDetailsByFirstNameLastNameAndType($firstname,$lastname,$type)
        {

            //database connection
            include("../db.php");

            //user data query
            $user_data_query = "SELECT userUniqueno,username,type,firstname,lastname,email,userStatus FROM `userLogin` WHERE `firstname`= '$firstname' and lastname='$lastname' and `type`= '$type'";
            try 
            {
                $user_data = null;
                $user_data_query_result=mysqli_query($conn,$user_data_query);
                if(mysqli_num_rows($user_data_query_result) >= 1)
                {
                    $user_data_row=mysqli_fetch_array($user_data_query_result,MYSQLI_ASSOC);
                    $user_data = $user_data_row;
                    return $user_data;
                }            

            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
                return null;
            }

            //user not found
            return null;
        }
        /*
            NAME : getMemberDataBaseFields
            TYPE : global static utility function
            DETAILS : get member database field value based field arrray list
            RETURN : return array of member value else null;
        */
        public static function getMemberDataBaseFields($client_id,$year,$member_id,$column_name_list = [] )
        {
            include("../db.php");
            
            if(count($column_name_list)==0)
            {
                $column_name_list_string='*';
            }
            else
            {
                $column_name_list_string = implode(",",$column_name_list);
            }
            
            $member_column_details = NULL;
            $member_records_query = "SELECT $column_name_list_string FROM HRA_member_".$client_id.'_'.$year." WHERE member_id = '$member_id'";

            //echo $member_records_query;
            $member_records_query_result=mysqli_query($conn,$member_records_query);
            while($member_row = mysqli_fetch_array($member_records_query_result,MYSQLI_ASSOC))
            { 
                $member_column_details = $member_row; 
            }

            return $member_column_details;
        }

        /*  PENDING IMPLEMENTATION
            NAME : getMemberDataBaseFieldsAdvanced
            TYPE : global static utility function
            DETAILS : get member database field value based field arrray list with custom sql query string
            RETURN : return array of member value else null;
        */
        public static function getMemberDataBaseFieldsAdvanced($client_id,$year,$member_id,$custom_sql_query,$column_name_list = [] )
        {
            include("../db.php");

            if($custom_sql_query=="" || $custom_sql_query==NULL) { return NULL;}

            if(count($column_name_list)==0)
            {
               return NULL;
            }
            else
            {
                $column_name_list_string = implode(",",$column_name_list);
            }
        
            $members_detail = NULL;
            $member_records_query_result=mysqli_query($conn,$custom_sql_query);
            while($member_row = mysqli_fetch_array($member_records_query_result,MYSQLI_ASSOC))
            { 
                $members_detail[] = $member_row; 
            }

            return $members_detail;
        }

        /*
            NAME : updateMemberDatabseFields
            TYPE : global static utility function
            DETAILS : update member table multiple column field values
            RETURN : true if success otherwise false
        */
        public static function updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list)
        {

            include("../db.php");
            $column_name_count = count($column_name_list);
            $column_value_count =count($column_value_list);
            if($column_name_count!=$column_value_count) return false;

            $column_name_and_value_string ="";
            for($column=0;$column<$column_name_count;$column++)
            {
                //check for null value 
                $column_value_null_formatted = 'NULL';
                if($column_value_list[$column])
                {
                    $column_value_null_formatted = "'$column_value_list[$column]'";
                }

                $column_name_and_value_string.= $column_name_list[$column]." = $column_value_null_formatted ";
                if($column<$column_name_count-1)
                {
                    $column_name_and_value_string.=",";
                }
            }
            
          $updated_member_query = "UPDATE HRA_member_".$client_id.'_'.$year." SET  $column_name_and_value_string WHERE `member_id` = '$member_id'";
            $updated_member_query_result = NULL;
            try 
            {
                $updated_member_query_result=mysqli_query($conn,$updated_member_query);
                $updated_row_count = mysqli_affected_rows($conn);
                if($updated_row_count)
                {
                    return true;
                }
                else
                {	
                    //no value value change detected
                    if($updated_member_query_result)
                    {
                        return true;
                    }
                    return false;
                }	
            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
                return false;
            }

        }

         /*
            NAME    : updateSchedulerCallHraInDatabase
            TYPE    : static internal Utiltiy function 
            DETAILS : 
        */
        public static function updateSchedulerCallHraInDatabase($client_id,$year,$member_id,$status,$date,$details,$disabled,$notification=null)
        {
            $call_hra_status_array = array(

                'status'=>$status,
                'date'=>$date,
                'details'=>$details,
                'disabled'=>$disabled,
                'notification'=>$notification
            );
            $column_name_list = ['schedulerCallHraStatus'];
            $column_value_list =[json_encode($call_hra_status_array)];
            return self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
        }

        /*
            NAME    : runUpdateSqlQuery
            TYPE    : global sql Utiltiy function 
            DETAILS : run UPDATE sql query 
        */
        public static function runUpdateSqlQuery($sql_query_string)
        {
            //database connectio
            include("../db.php");
            $sql_query_result = NULL;
            try 
            {
                $sql_query_result=mysqli_query($conn,$sql_query_string);
                $updated_row_count = mysqli_affected_rows($conn);
                if($updated_row_count)
                {
                    return true;
                }
                else
                {	
                    //no value value change detected
                    if($sql_query_result)
                    {
                         return true;
                    }

                    return false;
                }	
            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
                return false;
             }
                            

        }

        /*
            NAME    : updateHraStatus
            TYPE    : global  Utiltiy function 
            DETAILS :  update hra status 
        */
        public static function updateHraStatus($client_id,$year,$member_id,$hra_status,$update_if_active_hra_status=-1)
        {
            //updated the hra status to unscheduled
            
            $member_details = self::getMemberDataBaseFields($client_id,$year,$member_id,["hraStatus"]);
            $hra_status_array = json_decode($member_details["hraStatus"],true);
            $update_hra_status_string = NULL;
            
            if($hra_status_array && is_array($hra_status_array) && array_key_exists("hraStatus",$hra_status_array))
            {
                if($update_if_active_hra_status==-1)
                {
                    $hra_status_array["hraStatus"] = $hra_status;
                    $update_hra_status_string = json_encode($hra_status_array);
                }
                else
                {
                    if($update_if_active_hra_status==$hra_status_array["hraStatus"])
                    {
                        $hra_status_array["hraStatus"] = $hra_status;
                        $update_hra_status_string = json_encode($hra_status_array);
                    }
                }
            }
            else
            {  
                $hra_status_string='{"hraStatus":'.$hra_status.',"subStatus":"","cancelStatus":"","cancelText":""}';
                $update_hra_status_string = $hra_status_string;  
            }
            

            if($update_hra_status_string)
            {
                $column_name_list =['hraStatus'];
                $column_value_list=[$update_hra_status_string];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
                return true;
            }
            
            return false;
        }

        /*
            NAME    : updateHraStatusWithCondition
            TYPE    : global  Utiltiy function 
            DETAILS :  update hra status and 
        */
        public static function updateHraStatusWithCondition($client_id,$year,$member_id,$hra_status_json,$new_hra_status,$update_if_hra_status_array=null)
        {
            //if hra json paramemter null the get the member hra status
            $hra_status_array = null;
            if($hra_status_json==null)
            {
                $member_details = self::getMemberDataBaseFields($client_id,$year,$member_id,["hraStatus"]);
                $hra_status_array = json_decode($member_details["hraStatus"],true);
            }
            else
            {
                //updated the hra status to unscheduled
                $hra_status_array = json_decode($hra_status_json,true);
            }
            
            $update_hra_status_string = NULL;
            
            if($hra_status_array && is_array($hra_status_array) && array_key_exists("hraStatus",$hra_status_array))
            {
                if($update_if_hra_status_array==null)
                {
                    $hra_status_array["hraStatus"] = $new_hra_status;
                    $update_hra_status_string = json_encode($hra_status_array);
                }
                else
                {
                    if(is_array($update_if_hra_status_array))
                    {
                        if(in_array($hra_status_array["hraStatus"],$update_if_hra_status_array))
                        {
                            $hra_status_array["hraStatus"] = $new_hra_status;
                            $update_hra_status_string = json_encode($hra_status_array);
                        }
                    }
                }
            }
            else
            {  
                $hra_status_string='{"hraStatus":'.$new_hra_status.',"subStatus":"","cancelStatus":"","cancelText":""}';
                $update_hra_status_string = $hra_status_string;  
            }
            

            if($update_hra_status_string)
            {
                $column_name_list =['hraStatus'];
                $column_value_list=[$update_hra_status_string];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
                return true;
            }
            
            return false;
        }

        /*
            NAME    : updateSchedulerCallHraStatusWithCondition
            TYPE    : global Utiltiy function 
            DETAILS :  update scheduler hra status with condition of what previous or nothing or null 
        */
        public static function updateSchedulerCallHraStatusWithCondition($client_id,$year,$member_id,$new_scheduler_hra_status_array,$update_if_status_array=null)
        {
            //if hra json paramemter null the get the member hra status
            $scheduler_hra_status_array = null;
            
            $member_details = self::getMemberDataBaseFields($client_id,$year,$member_id,["schedulerCallHraStatus"]);
            $scheduler_hra_status_array = json_decode($member_details["schedulerCallHraStatus"],true);
            
            $update_scheduler_hra_status_string = null;
            
            if($scheduler_hra_status_array && is_array($scheduler_hra_status_array) && array_key_exists("status",$scheduler_hra_status_array))
            {
                if($update_if_status_array==null)
                {
                    $update_scheduler_hra_status_string = json_encode($new_scheduler_hra_status_array);
                }
                else
                {
                    if(is_array($update_if_status_array))
                    {
                        if(in_array($scheduler_hra_status_array["status"],$update_if_status_array))
                        {
                            $update_scheduler_hra_status_string = json_encode($new_scheduler_hra_status_array);
                        }
                    }
                }
            }
            else
            {  
                $update_scheduler_hra_status_string = json_encode($new_scheduler_hra_status_array); 
            }
            

            if($update_scheduler_hra_status_string)
            {
                $column_name_list =['schedulerCallHraStatus'];
                $column_value_list=[$update_scheduler_hra_status_string];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
                return true;
            }
            
            return false;
        }

        /*
            NAME    : insertMemberCityIfNotAvailable
            TYPE    : global Utiltiy function 
            DETAILS :  
        */
        public static function insertMemberCityIfNotAvailable($city_name,$state_name)
        {
            include("../db.php");
            
            $city_name = StringUtility::trimAndRemoveSpecialCharactersExceptSpace($city_name);
            $state_name = StringUtility::trimAndRemoveSpecialCharactersExceptSpace($state_name);

            $city_name=strtoupper($city_name);
            $state_name=strtoupper($state_name);

            $member_city_insert_query = $conn->prepare('INSERT IGNORE INTO `HRA_member_city`(`name`, `state`) VALUES (?,?)');
            $member_city_insert_query->bind_param("ss",$city_name,$state_name);
            $member_city_insert_query->execute();
            if($member_city_insert_query->affected_rows>0)
			{	
                return true;
            }

            return false;
        }

        /*
            NAME    : updateScheduleDate
            TYPE    : global Utiltiy function 
            DETAILS :  
        */
        public static function updateScheduleDate($client_id,$year,$member_id,$schedule_date)
		{
                $column_name_list =['scheduleDate'];
                $column_value_list=[$schedule_date];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
		}  


         /*
            NAME    : updateVisitType
            TYPE    : global Utiltiy function 
            DETAILS :  
        */
        public static function updateVisitType($client_id,$year,$member_id,$visit_type)
		{
            $formatted_visit_type  = trim(strtolower($visit_type));

            if($formatted_visit_type == 'virtual')
            {
                $column_name_list =['VisitType'];
                $column_value_list=['Virtual'];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
            }
            else if($formatted_visit_type == 'in person')
            {
                $column_name_list =['VisitType'];
                $column_value_list=['In Person'];
                self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
            }
            else
            {

            }
                
		} 

        /*
            NAME    : updateMemberAvailForScheduleStatus
            TYPE    : global  Utility Function
            DETAILS : update the members availForSchedule flag value if following hra status
            RETURN  : true updated else false ( if no change in data update also return false)
        */
        public static function updateMemberAvailForScheduleStatus($client_id,$year,$member_id,$availForSchedule,$hra_status_json,$update_if_hra_status_array)
        {
            include("../db.php");

            $is_hra_status_match = false;
            $hra_status_array = json_decode($hra_status_json,true);
            if($hra_status_array && is_array($hra_status_array) && array_key_exists("hraStatus",$hra_status_array))
            {
                if($update_if_hra_status_array==null)
                {
                    $is_hra_status_match = false;
                }
                else
                {
                    if(is_array($update_if_hra_status_array))
                    {
                        if(in_array($hra_status_array["hraStatus"],$update_if_hra_status_array))
                        {
                            $is_hra_status_match = true;
                            
                        }
                    }
                }
            }

            if($is_hra_status_match==true)
            {

                try 
                {
                    $update_member_sql_query = "UPDATE HRA_member_".$client_id.'_'.$year." SET availForSchedule = '$availForSchedule' WHERE member_id = '$member_id'";
                    $member_updated_result = mysqli_query($conn,$update_member_sql_query);
                    $is_updated = mysqli_affected_rows($conn);
                    //echo " client_id:".$client_id.",database updated:".$is_updated;
                    if($is_updated)
                    {
                        return true;
                    }

                    return false;

                }
                catch (Exception $e)
                {
                    $error = $e->getMessage();
                    return false;
                }
            }
            return false;
        }

    /*
        NAME    : updateMemberScheduleStatus
        TYPE    : global  Utility Function
        DETAILS : update the members availForSchedule flag value
        RETURN  : true updated else false ( if no change in data update also return false)
    */
    public static function updateMemberScheduleStatus($client_id,$year,$member_id,$availForSchedule)
    {
        include("../db.php");
        //database connection
        $connection = $conn;
        try 
        {
            $update_member_sql_query = "UPDATE HRA_member_".$client_id.'_'.$year." SET availForSchedule = '$availForSchedule' WHERE member_id = '$member_id'";
            $member_updated_result = mysqli_query($connection,$update_member_sql_query);
            $is_updated = mysqli_affected_rows($connection);
            //echo " client_id:".$client_id.",database updated:".$is_updated;
            if($is_updated)
            {
                return true;
            }

            return false;

        }
        catch (Exception $e)
		{
			$error = $e->getMessage();
			return false;
        }
        
        return false;
    }

    /* by Suresh
        NAME    : updateEvalutionDate
        TYPE    : global Utiltiy function 
        DETAILS :  Update Evalaution date based on memberid
    */
    public static function updateEvalutionDate($client_id,$year,$member_id,$evalution_date)
    {
        $formatted_evalution_date  = trim($evalution_date);

        if($formatted_evalution_date != '' && $formatted_evalution_date != 'undefinied'  && $formatted_evalution_date != null)
        {
            $column_name_list =['EvaluationDate'];
            $column_value_list=[$formatted_evalution_date];
            self::updateMemberDatabseFields($client_id,$year,$member_id,$column_name_list,$column_value_list);
        }
        else
        {

        }
            
    } 

}

?>